package net.kldp.j2ee.kupload;

import java.io.IOException;
import java.io.UnsupportedEncodingException;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import static net.kldp.j2ee.kupload.HeaderParser.*;

/**
 * MultiPart Upload 처리 클래스
 * 
 */
public class HttpServletUpload {
	private UploadFiles uploadFiles = new UploadFiles();
	private Parameters parameters = new Parameters();
	private String encoding = "utf-8";
	private final int MEMORY_SIZE = 1024 * 1024 * 2;
	/**
	 * MultiPart Upload 처리 클래스를 생성한다.
	 * 
	 * @param request
	 * @throws HttpServletUploadException
	 * @throws IOException
	 * @throws ServletException
	 */
	public HttpServletUpload(HttpServletRequest request) throws HttpServletUploadException, IOException,
			ServletException {
		execute(request);
	}

	/**
	 * MultiPart Upload 처리 클래스를 생성한다.
	 * 
	 * @param request
	 * @param encoding
	 * @throws HttpServletUploadException
	 * @throws IOException
	 * @throws ServletException
	 */
	public HttpServletUpload(HttpServletRequest request, String encoding) throws HttpServletUploadException,
			IOException, ServletException {
		this.encoding = encoding;
		execute(request);
	}

	/**
	 * 스트림을 파싱하여 파일업로드와 파라미터를 처리한다.
	 * 
	 * @param request
	 * @throws IOException
	 * @throws HttpServletUploadException
	 */
	private void execute(HttpServletRequest request) throws IOException, HttpServletUploadException {
		String dataHeader;
		DataSection section = new DataSection(request.getInputStream(), request.getContentLength(),
				MEMORY_SIZE);
		while (seekNextDataSection(section)) {
			dataHeader = getDataHeader(section);
			seekBinaryData(section);
			if (isFile(dataHeader))
				addFile(section, dataHeader);
			else
				parameters.putParameter(getFieldName(dataHeader), getFieldValue(section));
		}
	}

	/**
	 * 업로드된 바이너리 파일을 얻는다.
	 * 
	 * @return uploadFiles
	 */
	public UploadFiles getUploadFiles() {
		return uploadFiles;
	}

	/**
	 * Form의 field 파라미터를 얻는다.
	 * 
	 * @return parameters
	 */
	public Parameters getRequestParameters() {
		return parameters;
	}

	/**
	 * 다음 데이터 섹션을 검색한다.
	 * 
	 * @param section
	 * @return boolean
	 */
	private boolean seekNextDataSection(DataSection section) {
		if (section.position == 0)
			return seekFirstDataSection(section);
		else if (section.buffer.get(section.position + 1) == 45)
			return false;
		else if (section.position + 2 >= section.length) {
			section.position += 2;
			return false;
		} else
			return true;
	}

	/**
	 * 스트림에서 첫번째 데이터 섹션을 찾는다.
	 * 
	 * @param section
	 */
	private boolean seekFirstDataSection(DataSection section) {
		byte b;
		while (section.position < section.length) {
			b = section.buffer.get(section.position);
			if (b == 13)
				break;
			else
				section.boundary.append((char) b);
			section.position++;
		}
		section.position++;
		if (section.position == 2)
			return false;
		return true;
	}

	/**
	 * 데이터 섹션에서 필드값을 추출한다.
	 * 
	 * @param section
	 * @return fieldValue
	 */
	private String getFieldValue(DataSection section) {
		return toString(section, section.start, section.end);
	}

	private void addFile(DataSection section, String dataHeader) {
		MappedDiskFile file;
		if (isMacBinary(dataHeader))
			section.start += 128;
		file = new MappedDiskFile(section.buffer, section.start, section.end);
		file.setFieldName(getFieldName(dataHeader));
		file.setContentType(getContentType(dataHeader));
		file.setFilePath(getFilePath(dataHeader));
		uploadFiles.putFile(file.getFieldName(), file);
	}

	/**
	 * 스트림에서 데이터섹션을 찾는다.
	 * 
	 * @param section
	 */
	private void seekBinaryData(DataSection section) {
		int searchPosition = section.position;
		int keyPosition = 0;
		section.start = section.position;
		while (searchPosition <= section.length) {
			if (section.buffer.get(searchPosition) == (byte) section.boundary.charAt(keyPosition)) {
				if (keyPosition == section.boundary.length() - 1) {
					section.end = searchPosition - section.boundary.length() - 2;
					break;
				}
				keyPosition++;
			} else {
				keyPosition = 0;
			}
			searchPosition++;
		}
		section.position = section.end + section.boundary.length() + 3;
	}

	/**
	 * 섹션에서 데이터헤더를 분리해낸다.
	 * 
	 * @param section
	 * @return dataHeader
	 */
	private String getDataHeader(DataSection section) {
		int start = section.position;
		int end = 0;
		while (true) {
			if (section.buffer.get(section.position) == 13 && section.buffer.get(section.position + 2) == 13) {
				end = section.position - 1;
				section.position += 2;
				break;
			} else {
				section.position++;
			}
		}
		section.position += 2;
		return toString(section, start, end);
	}

	/**
	 * ByteBuffer에서 문자열을 추출해낸다.
	 * 
	 * @param section
	 * @param start
	 * @param end
	 * @return String
	 */
	private String toString(DataSection section, int start, int end) {
		byte[] b = new byte[(end - start) + 1];
		section.buffer.position(start);
		section.buffer.get(b, 0, (end - start) + 1);
		try {
			return new String(b, encoding);
		} catch (UnsupportedEncodingException e) {
		}
		return null;
	}

	@Override
	protected void finalize() throws Throwable {
		super.finalize();
		uploadFiles.finalize();
		uploadFiles = null;
		parameters = null;
	}
}